## Present basic time series data for Helio output or Fq' / Fm'
## Author: Jim Stevens
## Date: 10/04/2024
## Version 4.3.3

#### Housekeeping ####
rm(list=ls())

library(tidyverse)
library(openxlsx)
library(lubridate) # create dd mm yyyy time
library(hms) # create hh mm ss time
library(scales) # to operate date plotting in ggplot
library(cowplot) # to combine figures

#### read in data

# fluctuating

tmp0 <- read.xlsx("./Data/Feedback System Data/1122FEEDBACK_COMBINED.xlsx",
                  detectDates=TRUE) # chl f data
tmp1_h <- read.xlsx("./Data/Feedback System Data/Feedback_fluctuating_data_combined.xlsx",
                    detectDates=TRUE)
# square wave
tmp00 <- read.xlsx("./Data/Feedback System Data/PSP_Data_1123_stable_combined.xlsx",
                  detectDates=TRUE)

# cumulative power meter
tmp_cpm <- read.xlsx("./Data/Feedback System Data/Cumulative_Power_Use.xlsx")


str(tmp0)
str(tmp1_h)
str(tmp00) # data are in slightly different format; tmp3 dealt with after tmp1
str(tmp_cpm)

#### Fluctuating ####

#### 1. Get the chl f probe data working ####

tmp0$exp <- as.factor(tmp0$exp)
tmp0$Time <- tmp0$Time*(24*60*60) # turn tmp1$Time from a fraction to 
# number of seconds in 1 day
tmp0$Time <- as_hms(tmp0$Time) # convert to hms data type

tmp1 <- tmp0 %>%
  filter(exp!=3|day!=5) %>% # nonsense data
  filter(exp!=3|day!=-45132) %>% # nonsense data
  filter((Fs>200|Fs<0) %>% # eliminate values where probe is in error
  replace_na(T)) %>%
  mutate(new_t=case_when(
    exp==3 ~ as_hms(as.numeric(Time)-7200), # subtract 2 hours to align for dawn
    exp==6 ~ as_hms(as.numeric(Time)-7200), # subtract 2 hours to align for dawn
    TRUE ~ Time
  )) %>% # align the times for all experiments to start at 1000 and end at 0400
  mutate(nt=case_when(
    new_t<0 ~ as_hms(new_t+86400),
    new_t>0 ~ as_hms(new_t) # reset clock to correct time
  ))

tmp1$Date <- as.POSIXct("2024-01-01")
tmp1$date_time <- ymd_hms(paste(tmp1$Date, tmp1$nt))

lookat <- c("FqpFmp"="Y(II)","FqpFvp"="Fq'/Fv'","FvpFmp"="Fv'Fm'","FvFm"="Fv/Fm")
tmp1 <- tmp1 %>%
  rename(all_of(lookat)) # get some sensible names

## Now create 15 minute bins to combine data
min_time <- min(tmp1$date_time) - difftime(min(tmp1$date_time), 
                                           as.Date(min(tmp1$date_time))) 

breaks <- seq.POSIXt(min_time, max(tmp1$date_time)+15*60, by="15 min") # create POSIX object with breaks every `15 minutes throughout a day
tmp1$bins <- cut(tmp1$date_time, breaks=breaks)
# bin all the data in tmp1 into 15 minute bins

# Calculate values for Chl F parameters
tmp2 <- tmp1 %>%
    group_by(bins,exp,day) %>%
    filter(exp!=1|day!=1) %>% # day 1 expt 1 is nonsense
    summarise(N=n(),FqFm=mean(FqpFmp,na.rm=T), # avg @ expt level
              FqFv=mean(FqpFvp,na.rm=T),
              FvFm=mean(FvpFmp,na.rm=T)
              )

# create bin for NPQ calc
tnpq <- tmp1 %>%
  filter(Type=="Fv/Fm") %>%
  group_by(exp,day) %>%
  filter(bins=="2024-01-01 09:45:00"|bins=="2024-01-01 09:30:00") %>%
  filter(FvFm>0.75) %>% # get rid of nonsense Fv/Fm values
  summarise(avg=mean(FvFm))# %>%
  #group_by(exp) %>%
  #summarise(avg=mean(avg),se=sqrt((sum(se)^2)/n() ))

# add correct Fv/Fm values for NPQ calc
tmp2 <- left_join(tmp2,tnpq,by=c("exp","day")) %>%
  group_by(bins,exp) %>%
  reframe(N=n(),mean_FqFm=mean(FqFm),se_FqFm=sd(FqFm)/sqrt(n()), # avg @ expt level
            mean_FqFv=mean(FqFv),se_FqFv=sd(FqFv)/sqrt(n()),
            mean_FvFm=mean(FvFm),se_FvFm=sd(FvFm)/sqrt(n()),
            mean_NPQ=avg/FvFm-1,se_NPQ=sd((avg/FvFm-1))/sqrt(n())
  ) %>%
 group_by(bins) %>%
  summarise(N=n(),mean_FqFm=mean(mean_FqFm,na.rm=T),se_FqFm=mean(se_FqFm,na.rm=T), # avg @ expt level
            mean_FqFv=mean(mean_FqFv,na.rm=T),se_FqFv=mean(se_FqFv,na.rm=T),
            mean_FvFm=mean(mean_FvFm,na.rm=T),se_FvFm=mean(se_FvFm,na.rm=T),
            mean_NPQ=mean(mean_NPQ,na.rm=T),se_NPQ=mean(se_NPQ,na.rm=T)
  ) %>%
  mutate(mean_FqFm=replace_na(mean_FqFm,0), se_FqFm=replace_na(se_FqFm,0),
         mean_FqFv=replace_na(mean_FqFv,0), se_FqFv=replace_na(se_FqFv,0),
         mean_FvFm=replace_na(mean_FvFm,0), se_FvFm=replace_na(se_FvFm,0),
         mean_NPQ=replace_na(mean_NPQ,0), se_NPQ=replace_na(se_NPQ,0)
  ) %>%
  na.omit()

str(tmp2)
# create plottable object of mean and se for all 3 measurements
tmp2$DTB <- as_datetime(ymd_hms(tmp2$bins)) 
# create new non-factor value for bins
tmp2$DTB[1] <- make_datetime(2024,01,01,00,00,00)
# the first element is now null, so replace it with midnight
tmp2$DTB <- tmp2$DTB+hours(14) # Get 10:00 to 00:00 as start point
#tmp2$DTB[9] <- make_datetime(2024,01,01,14,00,00)

tmp2$DTA <- as_hms(tmp2$DTB) # separate time from date
#tmp2$DTA[9] <- hms(second(tmp2$DTA[9]),minute(tmp2$DTA[9]),hour(tmp2$DTA[9])+2)
tmp2$DT <- as.POSIXct(ymd("2024-01-01")) # create a date placeholder
tmp2$DTC <- as.POSIXct(paste(tmp2$DT,tmp2$DTA)) # combine the two

#### 2. Get the heliospectra output working ####

lookup <- c(Date="dd/mm/yy",Time="hh:mm:ss",Dawn_Dusk="Dawn/Dusk") 
# create some new names
tmp1_h <- tmp1_h %>%
  rename(all_of(lookup)) # rename some weird columns

tmp1_h$expt.no <- as.factor(tmp1_h$expt.no)
tmp1_h$Time <- tmp1_h$Time*(24*60*60) # turn tmp1$Time from a fraction to 
# number of seconds in 1 day
tmp1_h$Time <- round_hms(as_hms(tmp1_h$Time),digits=0) # convert to hms data type

tmp1_h$Time <- as_hms(tmp1_h$Time) # convert to hms data type

tmp1_h$Date <- as.POSIXct("2024-01-01")
tmp1_h$date_time <- ymd_hms(paste(tmp1_h$Date, tmp1_h$Time))

## Now create 15 minute bins to combine data
min_time <- min(tmp1_h$date_time) - difftime(min(tmp1_h$date_time), 
                                           as.Date(min(tmp1_h$date_time))) 

breaks <- seq.POSIXt(min_time, max(tmp1_h$date_time)+15*60, by="15 min") 
# create POSIX object with breaks every `15 minutes throughout a day
tmp1_h$bins <- cut(tmp1_h$date_time, breaks=breaks)
# bin all the data in tmp1 into 15 minute bins

tmp2_h <- tmp1_h %>%
  group_by(bins,expt.no) %>%
  filter(Dawn_Dusk==1) %>%
  summarise(N=n(),mean_helio=mean(Output,na.rm=T),se_helio=sd(Output,na.rm=T)/sqrt(n())) %>%
  summarise(N=n(),mean_helio=mean(mean_helio),se_helio=sd(mean_helio)/sqrt(n()))
# create plottable object of mean and se for all 3 measurements
tmp2_h$se_helio <- 0

tmp2_h$DTB <- as_datetime(ymd_hms(tmp2_h$bins)) 
# create new non-factor value for bins
tmp2_h$DTB[1] <- make_datetime(2024,01,01,00,00,00)
# the first element is now null, so replace it with midnight
tmp2_h$DTB <- tmp2_h$DTB+hours(14) # Get 10:00 to 00:00 as start point

tmp2_h$DTA <- as_hms(tmp2_h$DTB) # separate time from date
tmp2_h$DT <- as.POSIXct(ymd("2024-01-01")) # create a date placeholder
tmp2_h$DTC <- as.POSIXct(paste(tmp2_h$DT,tmp2_h$DTA)) # combine the two

## 3. combine the binned helio and chl f probe data 
tmp2_c <- NULL
tmp2_c <- right_join(as.data.frame(tmp2),as.data.frame(tmp2_h),by="bins")

## Plot heliospectra output

a <- ggplot(tmp2_c,aes(x=DTC.x,y=mean_helio,
                     ymin=mean_helio-se_helio,ymax=mean_helio+se_helio)) +
  geom_point() +
  geom_errorbar() +
  theme_bw()+
  theme(panel.grid=element_blank(),
        axis.text.x = element_text(angle = 45,hjust=1),
        text=element_text(size=12))+
  ylab("Heliospectra setting (  per 1000 )") +
  xlab("Time of Day (hh:mm)")+
  scale_x_datetime(breaks=date_breaks("2 hours"),
                   labels=date_format("%H:%M")) +
  scale_y_continuous(breaks = seq(0,90, by=10))
a  

## The same for Fq' / Fm'

b <- ggplot(tmp2_c,aes(x=DTC.x,y=mean_FqFm,
                    ymin=mean_FqFm-se_FqFm,ymax=mean_FqFm+se_FqFm)) +
  geom_point() +
  geom_errorbar() +
  theme_bw()+
  theme(panel.grid=element_blank(),
        axis.text.x = element_text(angle = 45,hjust=1),
        text=element_text(size=12))+
  ylab("Fq' / Fm' ") +
  xlab("Time of Day (hh:mm)")+
  scale_x_datetime(breaks=date_breaks("2 hours"),
                   labels=date_format("%H:%M")) +
  scale_y_continuous(breaks = seq(-0.1,1, by=50))+
  geom_hline(mapping=aes(yintercept=0.52))
b

#### Square wave / Control / Controlled Treatment ####

tmp00$Time <- tmp00$Time*(24*60*60) # turn tmp3$Time from a fraction to 
# number of seconds in 1 day
tmp00$Time <- as_hms(tmp00$Time) # convert to hms data type

tmp00$Date <- as.POSIXct("2024-01-01")
tmp00$date_time <- ymd_hms(paste(tmp00$Date, tmp00$Time))

lookie <- c(FqpFmp="Y(II)", FqpFvp="Fq'/Fv'",FvpFmp="Fv'Fm'",FvFm="Fv/Fm")
tmp00 <- tmp00 %>%
  rename(all_of(lookie))

tmp3 <- tmp00 %>%
  mutate(new_t=case_when(
    exp_no==3 ~ as_hms(as.numeric(Time)-7200), # subtract 2 hours to align for dawn
    exp_no==4 ~ as_hms(as.numeric(Time)-7200), # subtract 2 hours to align for dawn
    TRUE ~ Time
  )) %>% # align the times for all experiments to start at 1000 and end at 0400
  filter((Fs>50|Fs<0) %>% # eliminate values where probe is in error
           replace_na(T)) %>%
  mutate(nt=case_when(
         new_t<0 ~ as_hms(new_t+86400),
         new_t>0 ~ as_hms(new_t)
  ))
  
tmp3$date_time <- ymd_hms(paste(tmp3$Date, tmp3$new_t))

## Now create 15 minute bins to combine data
min_time <- min(tmp3$date_time) - difftime(min(tmp3$date_time), 
                                           as.Date(min(tmp3$date_time))) 

breaks <- seq.POSIXt(min_time, max(tmp3$date_time)+15*60, by="15 min") 
# create POSIX object with breaks every `15 minutes throughout a day
tmp3$bins <- cut(tmp3$date_time, breaks=breaks)
# bin all the data in tmp3 into 15 minute bins

tmp4 <- tmp3 %>%
  group_by(bins,exp_no,Day) %>%
  filter(PAR>0) %>%
  filter(Day!=1|exp_no!=1) %>% # day 1 expt 1 is nonsense
  filter(exp_no!=4|bins!='2024-01-01 10:15:00'|Day!=1) %>% # remove some NANs
  filter(exp_no!=6|bins!='2024-01-01 10:30:00'|Day!=1) %>%
  filter(exp_no!=3|bins!='2024-01-01 14:00:00'|Day!=1) %>%
  filter(exp_no!=5|bins!='2024-01-01 14:30:00'|Day!=1) %>%
  summarise(N=n(),FqFm=mean(FqpFmp,na.rm=T), # avg @ expt level and time
            FqFv=mean(FqpFvp,na.rm=T),
            FvFm=mean(FvpFmp,na.rm=T)
            ) #%>%
  
# create bin for NPQ calc
tnp <- tmp3 %>%
  filter(Type=="Fv/Fm") %>%
  group_by(exp_no,Day) %>%
  filter(bins=="2024-01-01 09:45:00"|bins=="2024-01-01 09:30:00") %>%
 mutate(FvFm = case_when(
         FvFm > 0.75 ~ FvFm,
         FvFm <= 0.75 ~ 0.75,
         TRUE ~ 0.75)) %>% # get rid of nonsense Fv/Fm values
  summarise(avg=mean(FvFm))

tmp4 <- left_join(tmp4,tnp,by=c("exp_no","Day")) %>%
  group_by(bins,exp_no) %>%
  reframe(N=n(),mean_FqFm=mean(FqFm,rm.na=T),se_FqFm=sd(FqFm)/sqrt(n()), # avg @ expt x bins level
          mean_FqFv=mean(FqFv,rm.na=T),se_FqFv=sd(FqFv)/sqrt(n()),
          mean_FvFm=mean(FvFm,rm.na=T),se_FvFm=sd(FvFm)/sqrt(n()),
          mean_NPQ=mean(avg/FvFm-1,na.rm=T),se_NPQ=mean(sd((avg/FvFm-1)/sqrt(N)),na.rm=T)
  ) %>%
  group_by(bins) %>% # crunch expts x bins to bins
  reframe(D=n(),mean_FqFm=mean(mean_FqFm),se_FqFm=mean(se_FqFm), # avg @ expt level
            mean_FqFv=mean(mean_FqFv),se_FqFv=mean(se_FqFv),
            mean_FvFm=mean(mean_FvFm),se_FvFm=mean(se_FvFm),
            mean_NPQ=mean(mean_NPQ,na.rm=T),se_NPQ = mean(se_NPQ,na.rm=T)
  ) %>%
  mutate(mean_helio=case_when(
    mean_FqFm>0~71,
    mean_FqFm==0~0,
  )) 
tmp4$se_helio=0
tmp4[17,c(3:12)] <- 0 # timing errors corrected at 04:00 lights off

# create plottable object of mean and se for all 3 measurements
tmp4$DTB <- as_datetime(ymd_hms(tmp4$bins)) 
# create new non-factor value for bins
tmp4$DTB[c(1)] <- make_datetime(2024,01,01,00,00,00)
# the first element is now null, so replace it with midnight
tmp4$DTB <- as_datetime(as.numeric(tmp4$DTB)-36000) # Get 10:00 to 00:00 as start point
#tmp4$DTB[9] <- make_datetime(2024,01,01,14,00,00)

tmp4$DTA <- as_hms(tmp4$DTB) # separate time from date
tmp4$DT <- as.POSIXct(ymd("2024-01-01")) # create a date placeholder
tmp4$DTC <- as.POSIXct(paste(tmp4$DT,tmp4$DTA)) # combine the two

## Plot heliospectra output
c <- ggplot(tmp4,aes(x=DTC,y=mean_helio,
                  ymin=mean_helio-se_helio,ymax=mean_helio+se_helio)) +
  geom_point() +
  geom_errorbar() +
  theme_bw()+
  theme(panel.grid=element_blank(),
        axis.text.x = element_text(angle = 45,hjust=1),
        text=element_text(size=12))+
  ylab("Heliospectra setting (  per 1000 )") +
  xlab("Time of Day (hh:mm)")+
  scale_x_datetime(breaks=date_breaks("2 hours"),
                   labels=date_format("%H:%M")) +
  scale_y_continuous(breaks = seq(0,90, by=10))
c  

## The same for Fq' / Fm'

d <- ggplot(tmp4,aes(x=DTC,y=mean_FqFm,
                     ymin=mean_FqFm-se_FqFm,ymax=mean_FqFm+se_FqFm)) +
  geom_point() +
  geom_errorbar() +
  theme_bw()+
  theme(panel.grid=element_blank(),
        axis.text.x = element_text(angle = 45,hjust=1),
        text=element_text(size=12))+
  ylab("Fq' / Fm' ") +
  xlab("Time of Day (hh:mm)")+
  scale_x_datetime(breaks=date_breaks("2 hours"),
                   labels=date_format("%H:%M")) +
  scale_y_continuous(breaks = seq(-0.1,1, by=0.1))+
  geom_hline(mapping=aes(yintercept=0.52))
d

#### Combine plots onto single graph for Helio and Fq' / Fm' settings ####
tmp2_c$Treatment <- "Fluctuating"

tmp4$Treatment <- "Square Wave"
# slope = 5.38, intercept = 3.08 for 18-83 setting of 100-450 PAR
tmp2_c$PAR_mean <- tmp2_c$mean_helio*5.38+3.08
tmp2_c$PAR_se <- tmp2_c$se_helio*5.38
tmp4$PAR_mean <- tmp4$mean_helio*5.38+3.08
tmp4$PAR_se <- tmp4$se_helio*5.38

tmp2_n <- tmp2_c %>%
  dplyr::select(DTA.x,Treatment, 
                mean_helio,se_helio,
                mean_FqFm,se_FqFm,
                mean_FqFv,se_FqFv,
                mean_FvFm,se_FvFm,
                mean_NPQ,se_NPQ,
                PAR_mean,PAR_se
                )
looksie <-c(DTA="DTA.x")
tmp2_n <- tmp2_n %>% rename(all_of(looksie))

tmp4_n <- tmp4 %>%
  dplyr::select(DTA,Treatment,
                mean_helio,se_helio,
                mean_FqFm,se_FqFm,
                mean_FqFv,se_FqFv,
                mean_FvFm,se_FvFm,
                mean_NPQ,se_NPQ,
                PAR_mean,PAR_se
  )

### Combine both Treatments for light and Chl F data
comb <- rbind(tmp2_n,tmp4_n) # combined PAR and PhiII data over time
comb$DTA <- as.POSIXct(comb$DTA)
comb$Treatment <- factor(comb$Treatment)
#comb$Treatment <- fct_relevel(comb$Treatment, "Fluctuating")
#comb$Treatment <- relevel(comb$Treatment,ref="Square Wave")

#### Create final output plots ####

combAss <- ggplot(comb,aes(x=DTA,y=PAR_mean,
                           ymin=PAR_mean-PAR_se,
                           ymax=PAR_mean+PAR_se,
                           colour=Treatment)) +
  geom_point(data = comb %>% filter(Treatment== "Square Wave")) +
  geom_errorbar(data = comb %>% filter(Treatment== "Square Wave")) +
  geom_point(data = comb %>% filter(Treatment== "Fluctuating")) +
  geom_errorbar(data = comb %>% filter(Treatment== "Fluctuating")) +
  theme_bw()+
  theme(panel.grid=element_blank(),
        axis.text.x = element_text(angle=45,hjust=1),
        text=element_text(size=12))+
  ylab(expression(paste("PFD ( ",~mu,mol,~m^-2,s^-1," )"))) +
  xlab("Time ( 24 h clock )")+
  scale_x_datetime(breaks=date_breaks("2 hours"),
                   labels=date_format("%H")) +
  scale_y_continuous(breaks = seq(0,500, by=50))+
  scale_color_manual(values=c('darkgreen','goldenrod'),
                     labels=c("Controlled","Feedback"))
combAss

combChl <- ggplot(comb,aes(x=DTA,
                           y=mean_FqFm,
                           ymin=mean_FqFm-se_FqFm,ymax=mean_FqFm+se_FqFm,
                           colour=Treatment)) +
  geom_point(data = comb %>% filter(Treatment== "Square Wave"),size=1) +
  geom_errorbar(data = comb %>% filter(Treatment== "Square Wave")) +
  geom_point(data = comb %>% filter(Treatment== "Fluctuating"),size=1) +
  geom_errorbar(data = comb %>% filter(Treatment== "Fluctuating")) +
  theme_bw()+
  theme(legend.title=element_blank(),
        legend.text=element_text(size=8),
        panel.grid=element_blank(),
        axis.text.x = element_text(angle=45,hjust=1),
        text=element_text(size=12))+
  ylab(expression(italic("Fq'/Fm' "))) +
  xlab("Time ( 24 h clock )")+
  scale_x_datetime(breaks=date_breaks("2 hours"),
                   labels=date_format("%H")) +
  ylim(0.3,0.7)+
  #scale_y_continuous(breaks = seq(0.3,1, by=0.1,limits=c(0.3,0.7)))+
  geom_hline(mapping=aes(yintercept=0.52),linetype="dashed")+
  guides(colour = guide_legend(nrow = 1))+
  scale_color_manual(values=c('darkgreen','goldenrod'),
                     labels=c("Controlled","Feedback"))

combChl

## Now do qP (Fq'/Fv') ##

combqP <- ggplot(comb,aes(x=DTA,y=mean_FqFv,
                           ymin=mean_FqFv-se_FqFv,ymax=mean_FqFv+se_FqFv,
                           colour=Treatment)) +
  geom_point(data = comb %>% filter(Treatment== "Square Wave"),size=1) +
  geom_errorbar(data = comb %>% filter(Treatment== "Square Wave")) +
  geom_point(data = comb %>% filter(Treatment== "Fluctuating"),size=1) +
  geom_errorbar(data = comb %>% filter(Treatment== "Fluctuating")) +
  theme_bw()+
  theme(legend.title=element_blank(),
        legend.position="bottom",
        panel.grid=element_blank(),
        axis.text.x = element_text(),
        text=element_text(size=12))+
  ylab(expression(italic("Fq'/Fv' "))) +
  xlab("Time ( 24 h clock )")+
  scale_x_datetime(breaks=date_breaks("2 hours"),
                   labels=date_format("%H")) +
  ylim(0.6,0.95)+
  #scale_y_continuous(breaks = seq(0.3,1, by=0.1,limits=c(0.3,0.7)))+
  #geom_hline(mapping=aes(yintercept=0.52),linetype="dashed")+
  guides(colour = guide_legend(nrow = 1))+
  scale_color_manual(values=c('darkgreen','goldenrod'),
                     labels=c("Controlled","Feedback"))

combqP
#save_plot("./Outputs/Final Plots/Supplementary Information/S3_qP_Time_Series.png",combqP,
#          base_width=4)


## Now do max efficiency (Fv'/Fm') ##
combeff <- ggplot(comb,aes(x=DTA,y=mean_FvFm,
                          ymin=mean_FvFm-se_FvFm,ymax=mean_FvFm+se_FvFm,
                          colour=Treatment)) +
  geom_point(data = comb %>% filter(Treatment== "Square Wave"),size=1) +
  geom_errorbar(data = comb %>% filter(Treatment== "Square Wave")) +
  geom_point(data = comb %>% filter(Treatment== "Fluctuating"),size=1) +
  geom_errorbar(data = comb %>% filter(Treatment== "Fluctuating")) +
  theme_bw()+
  theme(legend.title=element_blank(),
        legend.position="bottom",
        panel.grid=element_blank(),
        axis.text.x = element_text(),
        text=element_text(size=12))+
  ylab(expression(italic("Fv'/Fm' "))) +
  xlab("Time ( 24 h clock )")+
  scale_x_datetime(breaks=date_breaks("2 hours"),
                   labels=date_format("%H")) +
  ylim(0.6,0.8)+
  #scale_y_continuous(breaks = seq(0.3,1, by=0.1,limits=c(0.3,0.7)))+
  #geom_hline(mapping=aes(yintercept=0.52),linetype="dashed")+
  guides(colour = guide_legend(nrow = 1))+
  scale_color_manual(values=c('darkgreen','goldenrod'),
                     labels=c("Controlled","Feedback"))

combeff
#save_plot("./Outputs/Final Plots/Supplementary Information/S4_Max_efficiency_Time_Series.png",combeff,
#          base_width=4)


## Now do NPQ (Fm/Fm'-1) ##

combNPQ <- ggplot(comb,aes(x=DTA,y=mean_NPQ,
                           ymin=mean_NPQ-se_NPQ,ymax=mean_NPQ+se_NPQ,
                           colour=Treatment)) +
  geom_point(data = comb %>% filter(Treatment== "Square Wave"),size=1) +
  geom_errorbar(data = comb %>% filter(Treatment== "Square Wave")) +
  geom_point(data = comb %>% filter(Treatment== "Fluctuating"),size=1) +
  geom_errorbar(data = comb %>% filter(Treatment== "Fluctuating")) +
  theme_bw()+
  theme(legend.title=element_blank(),
        legend.position="bottom",
        panel.grid=element_blank(),
        axis.text.x = element_text(),
        text=element_text(size=12))+
  ylab(expression(italic("NPQ "))) +
  xlab("Time ( 24 h clock )")+
  scale_x_datetime(breaks=date_breaks("2 hours"),
                   labels=date_format("%H")) +
  #ylim(0,0.05)+
  #scale_y_continuous(breaks = seq(0.3,1, by=0.1,limits=c(0.3,0.7)))+
  #geom_hline(mapping=aes(yintercept=0.52),linetype="dashed")+
  guides(colour = guide_legend(nrow = 1))+
  scale_color_manual(values=c('darkgreen','goldenrod'),
                     labels=c("Controlled","Feedback"))

combNPQ
save_plot("./Outputs/Final Plots/Supplementary Information/S3_NPQ_Time_Series.png",combNPQ,
          base_width=5)


## Now do ETR (0.5 x 0.8 x PAR x Fq'/Fm') ##

combETR <- ggplot(comb,aes(x=DTA,y=(mean_FqFm*0.84*0.5*PAR_mean),
                           ymin=(mean_FqFm-se_FqFm)*0.84*0.5*PAR_mean,
                           ymax=(mean_FqFm+se_FqFm)*0.84*0.5*PAR_mean,
                           colour=Treatment)) +
  geom_point() +
  geom_errorbar() +
  theme_bw()+
  theme(legend.title=element_blank(),
        legend.position="bottom",
        panel.grid=element_blank(),
        axis.text.x = element_text(),
        text=element_text(size=12))+
  ylab(expression(italic("ETR "))) +
  xlab("Time ( 24 h clock )")+
  scale_x_datetime(breaks=date_breaks("2 hours"),
                   labels=date_format("%H")) +
  ylim(0,150)+
  #scale_y_continuous(breaks = seq(0.3,1, by=0.1,limits=c(0.3,0.7)))+
  #geom_hline(mapping=aes(yintercept=0.52),linetype="dashed")+
  guides(colour = guide_legend(nrow = 1))+
  scale_color_manual(values=c('goldenrod','darkgreen'),
                     labels=c("Feedback","Controlled"))

combETR
save_plot("./Outputs/Final Plots/Supplementary Information/S4_ETR_Time_Series.png",combETR,
          base_width=5)

## Create new additional plot of cumulative power use ####

tmp_cpm$Experiment.Number <- as.factor(tmp_cpm$Experiment.Number)
tmp_cpm$Treatment <- as.factor(tmp_cpm$Treatment)
levels(tmp_cpm$Treatment) <- c("Controlled","Feedback")
str(tmp_cpm)

cpm1 <- tmp_cpm %>%
  group_by(Treatment,Day) %>%
  summarise(N=n(),Mean=mean(Power_use,na.rm=T),
            se=sd(Power_use,na.rm=T)/sqrt(N),.groups="drop")
str(cpm1)

cpm_plot <- ggplot(cpm1,aes(x=Day,y=Mean,
                            ymin=Mean-se,ymax=Mean+se,
                            colour=Treatment)) +
  geom_point() +
  geom_errorbar() +
  theme_bw()+
  theme(legend.title=element_blank(),
        legend.position="bottom",
        panel.grid=element_blank(),
        axis.text.x = element_text(),
        text=element_text(size=12)) +
  xlab("Experiment Duration ( Days )")+
  ylab("Cumulative Energy Use ( kW h )")+
  #guides(colour = guide_legend(nrow = 1))+
  scale_color_manual(values=c('darkgreen','goldenrod'),
                     labels=c("Controlled","Feedback"))
cpm_plot

#### create combined plot for chl f data ####
legend <- get_legend(combChl)

final1 <- plot_grid(combAss+theme(legend.position="none",
                                  axis.title=element_text(size=8)),
                    combChl+theme(legend.position="none",
                                  axis.title=element_text(size=8)),
                    cpm_plot+theme(legend.position="none",
                                   axis.title=element_text(size=8)),
                    align="hv",
                    labels=c("A","B","C",""),
                    nrow=1)
final1
final2 <- plot_grid(final1,legend,
                    rel_heights=c(0.8,0.1),nrow=2)
final2
save_plot("./Outputs/Final Plots/Fig_7_A__Chl_F_kWh_Time_Series.png",final2,
          base_width=8,base_height=3)

## Create combined plot for qP / max eff data ##
qpme <- plot_grid(combqP+theme(legend.position="none"),
                  combeff+theme(legend.position="none"),
                  align="hv",labels="AUTO")
qpme
qpme1 <- plot_grid(qpme,legend,
                   rel_heights=c(0.8,0.1),nrow=2)
qpme1
save_plot("./Outputs/Final Plots/Fig_8_qP_and_Max_Eff_Time_Series.png",qpme1,
          base_width=8)

#### Compare differences in outputs of Q and chl f ####

## DLI data
DLI <- comb %>% # calculate DLI for each treatment
  group_by(Treatment) %>%
  na.omit() %>%
  summarise(Sum_PAR=sum(PAR_mean)*15*60/1000000)
DLI
capture.output(DLI,
               file="./Outputs/Final Plots/Stats/Trial_DLI.txt")

Hourly <- comb %>% # cut down data to reveal hourly values only 
  group_by(Treatment) %>%
  filter(minute(DTA)==0) %>%
  relocate(DTA,Treatment) %>%
  na.omit()
  #dplyr::select(!c(1,2,3,4,7,9,10))

write.csv(Hourly,
          file="./Outputs/Final Plots/Stats/Trial_Hourly.csv")



## crunch down tmp1 and tmp3 to 1 value for each expt for each time point
## then compare at 3 and 9 hours from apparent dawn.

df.f <- tmp1_h %>%
  group_by(expt.no,bins) %>%
  filter(bins=="2024-01-01 13:00:00"|bins=="2024-01-01 01:00:00") %>%
  dplyr::select(expt.no,bins,Output,FqFm) %>%
  mutate(Output=Output*5.38+3.08) %>%
  mutate(FqFm=FqFm/1024) %>%
  summarise(Output=mean(Output),FqFm=mean(FqFm),.groups="drop")

df.c <- tmp3 %>%
  group_by(exp_no,bins) %>%
  filter(bins=="2024-01-01 13:00:00"|bins=="2024-01-01 01:00:00") %>%
  dplyr::select(exp_no,bins,FqpFmp) %>%
  mutate(Output=385) %>%
  rename("expt.no"="exp_no") %>%
  rename("FqFm"="FqpFmp") %>%
  summarise(Output=mean(Output),FqFm=mean(FqFm),.groups="drop")
df.c$expt.no <- as.factor(df.c$expt.no)

df.f$Treatment <- "feedback"
df.c$Treatment <- "control"
df <- full_join(df.c,df.f)

## test differences
mod.output.am <- lm(Output~Treatment,data=filter(df,bins=="2024-01-01 13:00:00"))
summary(mod.output.am)
anova(mod.output.am) # effect of treatment @ 1 pm on light intensity

mod.output.pm <- lm(Output~Treatment,data=filter(df,bins=="2024-01-01 01:00:00"))
summary(mod.output.pm)
anova(mod.output.pm) # effect of treatment at 1 am on light intensity

capture.output(summary(mod.output.am),
               file="./Outputs/Final Plots/Stats/Fig_3_PPFD_Diffs.txt")
capture.output(anova(mod.output.am),
               file="./Outputs/Final Plots/Stats/Fig_3_PPFD_Diffs.txt",
               append=T)
capture.output(summary(mod.output.pm),
               file="./Outputs/Final Plots/Stats/Fig_3_PPFD_Diffs.txt",
               append=T)
capture.output(anova(mod.output.pm),
               file="./Outputs/Final Plots/Stats/Fig_3_PPFD_Diffs.txt",
               append=T)


## difference in Fq' / Fm'
mod.fqfm.am <- lm(FqFm~Treatment,data=filter(df,bins=="2024-01-01 13:00:00"))
summary(mod.fqfm.am)
anova(mod.fqfm.am) # no diff at 1 pm

mod.fqfm.pm <- lm(FqFm~Treatment,data=filter(df,bins=="2024-01-01 01:00:00"))
summary(mod.fqfm.pm)
anova(mod.fqfm.pm) # effect of treatment at 1 am on Fq'/Fm'

capture.output(summary(mod.fqfm.am),
               file="./Outputs/Final Plots/Stats/Fig_3_FqFm_Diffs.txt")
capture.output(anova(mod.fqfm.am),
               file="./Outputs/Final Plots/Stats/Fig_3_FqFm_Diffs.txt",
               append=T)
capture.output(summary(mod.fqfm.pm),
               file="./Outputs/Final Plots/Stats/Fig_3_FqFm_Diffs.txt",
               append=T)
capture.output(anova(mod.fqfm.pm),
               file="./Outputs/Final Plots/Stats/Fig_3_FqFm_Diffs.txt",
               append=T)


## mean difference in total photons
df.sum <- df %>%
  group_by(expt.no,Treatment) %>%
  summarise(Output=mean(Output),.groups="drop") %>%
  mutate(Output=Output*((18*3600)/(1000000)))
mod.DLI <- lm(Output~Treatment,data=df.sum)
summary(mod.DLI)
anova(mod.DLI) # Higher DLI for Square Wave treatment
capture.output(summary(mod.DLI),
               file="./Outputs/Final Plots/Stats/Fig_3_DLI_Diffs.txt",
               append=TRUE)
capture.output(anova(mod.DLI),
               file="./Outputs/Final Plots/Stats/Fig_3_DLI_Diffs.txt",
               append=TRUE)

## difference in Fq'/Fv' eop
mod.fqfv.pm <- lm(FqFv~Treatment,data=filter(df,bins=="2024-01-01 01:00:00"))
summary(mod.fqfm.pm)
anova(mod.fqfm.pm) # effect of treatment at 1 am on Fq'/Fm'

capture.output(summary(mod.fqfm.am),
               file="./Outputs/Final Plots/Stats/Fig_3_FqFm_Diffs.txt")


### plot for shellie's methods figure ###
met1 <- ggplot(filter(comb,Treatment=="Fluctuating"),aes(x=DTA,y=PAR_mean,
                                                         ymin=PAR_mean-PAR_se,
                                                         ymax=PAR_mean+PAR_se)) +
  geom_line(colour="darkgoldenrod2",linewidth=3) +
  #geom_errorbar() +
  theme_bw()+
  theme(panel.grid=element_blank(),
        axis.text = element_blank(),
        axis.title=element_blank())+
  ylab(expression(paste("PPFD ( ",~mu,mol,~m^-2,s^-1," )"))) +
  xlab("Z Time ( 24 hour clock )")+
  scale_x_datetime(breaks=date_breaks("2 hours"),
                   labels=date_format("%H")) +
  scale_y_continuous(breaks = seq(0,500, by=50))
met1
ggsave("./Outputs/Shellie_fluctuating_plot.png",met1)

met2 <- ggplot(filter(comb,Treatment=="Square Wave"),aes(x=DTA,y=PAR_mean,
                                                         ymin=PAR_mean-PAR_se,
                                                         ymax=PAR_mean+PAR_se)) +
  geom_line(colour="darkgoldenrod2",linewidth=3) +
  #geom_errorbar() +
  theme_bw()+
  theme(panel.grid=element_blank(),
        axis.text = element_blank(),
        axis.title=element_blank())+
  ylab(expression(paste("PPFD ( ",~mu,mol,~m^-2,s^-1," )"))) +
  xlab("Z Time ( 24 hour clock )")+
  scale_x_datetime(breaks=date_breaks("2 hours"),
                   labels=date_format("%H")) +
  scale_y_continuous(breaks = seq(0,500, by=50))
met2
ggsave("./Outputs/Shellie_sq_wave_plot.png",met2)
